﻿using System;
using System.Runtime.InteropServices;
using SSH.CP.TmsClient.Misc;
using SSH.CP.TmsClient.Wraps;

namespace SSH.CP.TmsClient
{
	public abstract class TmsConnectionBase
	{
		#region Fields
		private readonly CallBackDelegate mInnerCallBackDelegate;
		private string mServer;
		private string mPipe;
		private string mUser;
		private Int32 mCid;
		private ServerTimeService mServerTimeService;
		#endregion

		#region Ctor
		public TmsConnectionBase()
		{
			mInnerCallBackDelegate = new CallBackDelegate(InnerCallbackHandler);
		}
		#endregion

		#region Props
		public string Server { get { return mServer; } set { mServer = value; } }
		public string Pipe { get { return mPipe; } set { mPipe = value; } }
		public string User { get { return mUser; } set { mUser = value; } }
		public Int32 Cid { get { return mCid; } }
		public ServerTimeService ServerTimeService
		{
			get
			{
				if (mServerTimeService == null)
					mServerTimeService = new ServerTimeService(this);
				return mServerTimeService;
			}
		}
		#endregion

		#region Virtual Methods

		protected abstract void OnCallback(int pSizeInBytes, IntPtr pBuffer);

		#endregion

		#region Public Methods

		public Int32 Open() // Ensure Cid Initialization 
		{
			if (mCid == 0)
			{
				mCid = Tmconn.tmcConnect(mServer, mPipe, mUser, mInnerCallBackDelegate, IntPtr.Zero);
			}
			return mCid;
		}

		public void Close()
		{
			if(mCid != 0)
			{
				Tmconn.tmcDisconnect(mCid);
				mCid = 0;
			}
		}

		public bool IsConnected()
		{
			return (mCid != 0) && (Tmconn.tmcConnState(mCid) == 0);
		}

		public void EnsureConnected()
		{
			if (mCid == 0)
			{
				Open();

				int errCode = Tmconn.tmcConnState(mCid);
				if (errCode != 0)
					throw new ConnectException("No connection to server TM", true);
			}
			else
			{
				int errCode = Tmconn.tmcConnState(mCid); // расходы???

				if(errCode != 0)
				{
					int cnt = Tmconn.tmcReconnectCount(mCid);
					if (cnt == 0)
						throw new ConnectException("No connection to server TM", true);
				}
			}
		}
		#endregion

		#region Private Methods

		private void InnerCallbackHandler(Int32 pSizeInBytes, IntPtr pBuffer, IntPtr pParam)
		{
			OnCallback(pSizeInBytes, pBuffer);
		}
		#endregion
	}

	public class TmsConnectionBase<TCallbackEventArgs> : TmsConnectionBase where TCallbackEventArgs : CallbackEventArgs
	{
		#region Fields
		#endregion

		#region Events
		public event EventHandler<TCallbackEventArgs> Callback;
		#endregion

		#region Ctor
		public TmsConnectionBase() : base()
		{
		}
		#endregion

		#region Props
		#endregion

		#region Virtual Methods

		protected override void OnCallback(int pSizeInBytes, IntPtr pBuffer)
		{
			if (pSizeInBytes > 0)
			{
				TCallbackEventArgs eventArgs = (TCallbackEventArgs)Activator.CreateInstance(typeof(TCallbackEventArgs));
				ParseCallback(eventArgs, pSizeInBytes, pBuffer);

				if (Callback != null)
					Callback(this, eventArgs);
			}
		}

		protected virtual void ParseCallback(TCallbackEventArgs pEventArgs, int pSizeInBytes, IntPtr pBuffer)
		{
			pEventArgs.Buffer = new byte[pSizeInBytes];
			Marshal.Copy(pBuffer, pEventArgs.Buffer, 0, pSizeInBytes);
		}

		#endregion

		#region Public Methods
		#endregion

		#region Private Methods
		#endregion
	}

	public class CallbackEventArgs : EventArgs
	{
		#region Fields
		private byte[] mBuffer;
		#endregion

		#region Ctor
		#endregion

		#region Props
		public byte[] Buffer { get { return mBuffer; } internal set { mBuffer = value; } }
		#endregion
	}
}