﻿using System;
using System.Runtime.InteropServices;
using SSH.CP.TmsClient.Wraps;

namespace SSH.CP.TmsClient
{
	public static class Utils
	{
		public static void WriteZero(IntPtr pPtr, int pSize)
		{
			for(int I=0; I<pSize; I++)
				Marshal.WriteByte(pPtr, I, 0);
		}

		public static bool FindDallasKey(out UInt32 key_type, out UInt32 port, out Int32 key_id, out string scratch_pad)
		{
			key_type = 2; //USB
			port = 0;

			if (Tmconn.KeyReadScratchPad(key_type, port, out key_id, out scratch_pad))
				return true;

			key_type = 0; //COM

			for (port = 1; port <= 4; port++ )
				if (Tmconn.KeyReadScratchPad(key_type, port, out key_id, out scratch_pad))
					return true;

			return false;
		}

		public static bool FindInterfaceKey()
		{
			UInt32 key_type;
			UInt32 port;
			Int32 key_id;
			string scratch_pad;

			if(!FindDallasKey(out key_type, out port, out key_id, out scratch_pad))
				return false;

			if (((key_id & 0x0ff) == 0x02) && String.CompareOrdinal(scratch_pad, 0, "Interface", 0, 9).Equals(0))
				return true;

			return false;
		}
	}

	public class MarshalHelper : IDisposable
	{
		private IntPtr mPtr1;

		public MarshalHelper() { }

		public MarshalHelper(int pNumInBytes)
			: this()
		{
			SetPtr1(pNumInBytes);
		}

		public IntPtr Ptr1 { get { return mPtr1; } }

		public void SetPtr1(int pNumInBytes)
		{
			SetPtr1(Marshal.AllocCoTaskMem(pNumInBytes));
		}

		public void SetPtr1(IntPtr pPtr)
		{
			mPtr1 = pPtr;
		}

		#region IDisposable Members

		public void Dispose()
		{
			if(mPtr1 != IntPtr.Zero)
			{
				Marshal.FreeCoTaskMem(mPtr1);
				mPtr1 = IntPtr.Zero;
			}
			
		}

		#endregion
	}

  public class StructArrayHelper<TStruct> : IDisposable
  {
    private int _capacity;
    private IntPtr _ptr;
    private int _itemSize;
    private Type _itemType;

    public StructArrayHelper(int capacity)
    {
      _itemType = typeof(TStruct);
      _itemSize = Marshal.SizeOf(_itemType);
      _capacity = capacity;
      _ptr = Marshal.AllocCoTaskMem(_itemSize * _capacity);
    }

    public int Capacity { get { return _capacity; } }
    public IntPtr Ptr { get { return _ptr; } }

    public IntPtr GetPtr(int index)
    {
      CheckIndex(index);
      return (IntPtr)((long)_ptr + (index * _itemSize));
    }

    public void Write(int index, TStruct item, bool destroyOld = false)
    {
      CheckIndex(index);
      var itemPtr = (IntPtr)((long)_ptr + (index * _itemSize));
      Marshal.StructureToPtr(item, itemPtr, destroyOld);
    }

    public TStruct Read(int index, bool destroy = false)
    {
      var itemPtr = (IntPtr)((long)_ptr + (index * _itemSize));
      var item = (TStruct)Marshal.PtrToStructure(itemPtr, _itemType);
      if (destroy)
        Marshal.DestroyStructure(itemPtr, _itemType);
      return item;
    }

    private void CheckIndex(int index)
    {
      if (index < 0 || index >= _capacity)
        throw new IndexOutOfRangeException();
    }
    public void Dispose()
    {
      if (_ptr != IntPtr.Zero)
      {
        Marshal.FreeCoTaskMem(_ptr);
        _ptr = IntPtr.Zero;
      }
    }
  }
}