﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Text;
using System.Xml;

namespace SSH.CP.Utils.Xml
{
	public static class XmlFns
	{
		public static XmlNode FindFirstElement(XmlDocument pDoc, string pElementName)
		{
			XmlNodeList nodeList = pDoc.GetElementsByTagName(pElementName);
			if (nodeList != null && (nodeList.Count > 0))
				return nodeList[0];
			return null;
		}

		public static XmlNode FindElement(XmlDocument pDoc, XmlNode pParentNode, string pElementName)
		{
			return pParentNode.SelectSingleNode(String.Format("//{0}", pElementName)); //???
		}

		#region Write Elements

		public static XmlNode AddElement(XmlDocument pDoc, XmlNode pParentNode, string pValueName, string pValue)
		{
			XmlNode node = pDoc.CreateElement(pValueName);
			pParentNode.AppendChild(node);
			//node.Value = pValue; //ERROR
			node.InnerText = pValue;
			return node;
		}

		public static XmlNode AddElement<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName, TValue pValue)
		{
			string value = Convert.ToString(pValue); //string value = pValue.ToString();
			return AddElement(pDoc, pParentNode, pValueName, value);
		}

		public static XmlNode CreateOrGetElement(XmlDocument pDoc, XmlNode pParentNode, string pElementName)
		{
			XmlNode node = pParentNode.SelectSingleNode(String.Format("//{0}", pElementName)); //???
			if (object.ReferenceEquals(node, null))
			{
				node = pDoc.CreateElement(pElementName);
				pParentNode.AppendChild(node);
			}
			return node;
		}

		public static XmlNode WriteElement(XmlDocument pDoc, XmlNode pParentNode, string pValueName, string pValue)
		{
			XmlNode node = CreateOrGetElement(pDoc, pParentNode, pValueName);
			//node.Value = pValue; //ERROR
			node.InnerText = pValue;
			return node;
		}

		public static XmlNode WriteElement<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName, TValue pValue)
		{
			string value = Convert.ToString(pValue); //string value = pValue.ToString();
			return WriteElement(pDoc, pParentNode, pValueName, value);
		}

		#endregion

		#region Write Attributes

		public static void AddAttributeToNode(XmlDocument xmlDoc, XmlNode node, string attributeName, string attributeValue)
		{
			if (!object.ReferenceEquals(node.Attributes[attributeName], null)) return;
			XmlAttribute newAttribute = xmlDoc.CreateAttribute(attributeName);
			newAttribute.Value = attributeValue;
			node.Attributes.Append(newAttribute);
		}

		public static void WriteAttribute(XmlDocument pDoc, XmlNode pNode, string pAttributeName, string pValue)
		{
			if (pNode == null) return;
			XmlNode node = pNode.Attributes.GetNamedItem(pAttributeName);
			if (node == null)
				AddAttributeToNode(pDoc, pNode, pAttributeName, pValue);
			else
				node.Value = pValue;
		}

		public static void WriteAttribute<TValue>(XmlDocument pDoc, XmlNode pNode, string pAttributeName, TValue pValue)
		{
			string value = Convert.ToString(pValue); //string value = pValue.ToString();
			WriteAttribute(pDoc, pNode, pAttributeName, value);
		}

		#endregion

		#region Read Elements

		public static string ReadElement(XmlNode pNode)
		{
			if (pNode == null) return null;
			return pNode.InnerText;
		}

		public static bool ReadElement<TValue>(XmlNode pNode, out TValue pValue)
		{
			string value = ReadElement(pNode);
			return TryConvert<TValue>(value, out pValue);
		}

		public static TValue ReadElement<TValue>(XmlNode pNode)
		{
			TValue value;
			ReadElement(pNode, out value);
			return value;
		}

		public static TValue ReadElement<TValue>(XmlNode pNode, TValue pDefaultValue)
		{
			TValue value;
			if (!ReadElement(pNode, out value))
				return pDefaultValue;
			return value;
		}

		#endregion

		#region Read Attributes

		public static string ReadAttribute(XmlNode pNode, string pAttributeName)
		{
			if (pNode == null) return null;
			XmlNode node = pNode.Attributes.GetNamedItem(pAttributeName);
			if (node == null) return null;
			return node.Value;
		}

		public static bool TryReadAttribute<TValue>(XmlNode pNode, string pAttributeName, out TValue pValue)
		{
			string value = ReadAttribute(pNode, pAttributeName);
			if (String.IsNullOrEmpty(value))
			{
				pValue = default(TValue);
				return false;
			}
			return TryConvert<TValue>(value, out pValue);
		}

		public static TValue ReadAttribute<TValue>(XmlNode pNode, string pAttributeName)
		{
			TValue value;
			TryReadAttribute(pNode, pAttributeName, out value);
			return value;
		}

		public static TValue ReadAttribute<TValue>(XmlNode pNode, string pAttributeName, TValue pDefaultValue)
		{
			TValue value;
			if (!TryReadAttribute(pNode, pAttributeName, out value))
				return pDefaultValue;
			return value;
		}

		#endregion

		#region Active Flag // Attribute "active"

		public static void WriteActiveFlag(XmlDocument pDoc, XmlNode pNode, bool pActive)
		{
			WriteAttribute(pDoc, pNode, "active", pActive ? "1" : "0");
		}

		public static bool ReadActiveFlag(XmlNode pNode)
		{
			if (!ReferenceEquals(pNode.Attributes["active"], null))
				if (DbValueConvert.ToInt(pNode.Attributes["active"].Value, 0) != 0)
					return true;
			return false;
		}

		#region ???
		public static string GetActiveAttributeValue(XmlNode pNode, string pAttributeName)
		{
			if (pNode == null) return null;
			if (ReferenceEquals(pNode.Attributes["active"], null)) return null;
			//----------
			int active = DbValueConvert.ToInt(pNode.Attributes["active"].Value, 0);
			if (active == 0) return null;
			//----------
			XmlNode node = pNode.Attributes.GetNamedItem(pAttributeName);
			if (node == null) return null;
			return node.Value;
		}

		public static string GetActiveElementValue(XmlNode pNode)
		{
			if (pNode == null) return null;
			if (ReferenceEquals(pNode.Attributes["active"], null)) return null;
			//----------
			int active = DbValueConvert.ToInt(pNode.Attributes["active"].Value, 0);
			if (active == 0) return null;
			//----------
			return pNode.InnerText;
		}

		public static bool TryGetActiveAttributeValue<TValue>(XmlNode pNode, string pAttributeName, out TValue pValue)
		{
			string value = GetActiveAttributeValue(pNode, pAttributeName);
			return TryConvert<TValue>(value, out pValue);
		}

		public static bool TryGetActiveElementValue<TValue>(XmlNode pNode, out TValue pValue)
		{
			string value = GetActiveElementValue(pNode);
			return TryConvert<TValue>(value, out pValue);
		}

		public static bool TryGetActiveElementValue(XmlNode pNode, out bool pValue)
		{
			string value = GetActiveElementValue(pNode);
			if (String.IsNullOrEmpty(value))
			{
				pValue = false;
				return false;
			}
			pValue = DbValueConvert.ToBool(value, false);
			return true;
		}

		public static bool TryGetActiveElementValue(XmlNode pNode, out int pValue)
		{
			string value = GetActiveElementValue(pNode);
			if (String.IsNullOrEmpty(value))
			{
				pValue = 0;
				return false;
			}
			pValue = DbValueConvert.ToInt(value, 0);
			return true;
		}
		#endregion

		#endregion

		#region Parent Node Values As Elements With "value" Attributes

		public static XmlNode AddValue(XmlDocument pDoc, XmlNode pParentNode, string pValueName, string pValue)
		{
			XmlNode node = pDoc.CreateElement(pValueName);
			pParentNode.AppendChild(node);
			WriteAttribute(pDoc, node, "value", pValue);
			return node;
		}

		public static XmlNode AddValue<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName, TValue pValue)
		{
			string value = Convert.ToString(pValue); //string value = pValue.ToString();
			return AddValue(pDoc, pParentNode, pValueName, value);
		}

		public static XmlNode AddValueAndFlag(XmlDocument pDoc, XmlNode pParentNode, string pValueName, string pValue, bool pActive)
		{
			XmlNode node = pDoc.CreateElement(pValueName);
			pParentNode.AppendChild(node);
			WriteActiveFlag(pDoc, node, pActive);
			WriteAttribute(pDoc, node, "value", pValue);
			return node;
		}

		public static XmlNode AddValueAndFlag<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName, TValue pValue, bool pActive)
		{
			string value = Convert.ToString(pValue); //string value = pValue.ToString();
			return AddValueAndFlag(pDoc, pParentNode, pValueName, value, pActive);
		}

		public static XmlNode WriteValue(XmlDocument pDoc, XmlNode pParentNode, string pValueName, string pValue)
		{
			XmlNode node = CreateOrGetElement(pDoc, pParentNode, pValueName);
			WriteAttribute(pDoc, node, "value", pValue);
			return node;
		}

		public static XmlNode WriteValue<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName, TValue pValue)
		{
			string value = Convert.ToString(pValue); //string value = pValue.ToString();
			return WriteValue(pDoc, pParentNode, pValueName, value);
		}

		public static bool ReadValue(XmlDocument pDoc, XmlNode pParentNode, string pValueName, out XmlNode pValueNode, out string pValue)
		{
			pValueNode = FindElement(pDoc, pParentNode, pValueName);
			if (pValueNode == null)
			{
				pValue = null;
				return false;
			}
			pValue = ReadAttribute(pValueNode, "value");
			return true;
		}

		public static bool ReadValue<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName, out XmlNode pValueNode, out TValue pValue)
		{
			string value;
			if (!ReadValue(pDoc, pParentNode, pValueName, out pValueNode, out value))
			{
				pValue = default(TValue);
				return false;
			}
			return TryConvert<TValue>(value, out pValue);
		}

		public static string ReadValue(XmlDocument pDoc, XmlNode pParentNode, string pValueName)
		{
			string value;
			XmlNode node;
			ReadValue(pDoc, pParentNode, pValueName, out node, out value);
			return value;
		}

		public static TValue ReadValue<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName)
		{
			TValue value;
			XmlNode node;
			ReadValue(pDoc, pParentNode, pValueName, out node, out value);
			return value;
		}

		public static bool ReadValueAndFlag(XmlDocument pDoc, XmlNode pParentNode, string pValueName, out XmlNode pValueNode, out string pValue, out bool pActive)
		{
			pValueNode = FindElement(pDoc, pParentNode, pValueName);
			if (pValueNode == null)
			{
				pValue = null;
				pActive = false;
				return false;
			}
			pValue = ReadAttribute(pValueNode, "value");
			pActive = ReadActiveFlag(pValueNode);
			return true;
		}

		public static bool ReadValueAndFlag<TValue>(XmlDocument pDoc, XmlNode pParentNode, string pValueName, out XmlNode pValueNode, out TValue pValue, out bool pActive)
		{
			string value;
			if (!ReadValueAndFlag(pDoc, pParentNode, pValueName, out pValueNode, out value, out pActive))
			{
				pValue = default(TValue);
				return false;
			}
			return TryConvert<TValue>(value, out pValue);
		}

		#endregion

		#region Special Collection: Fields

		public static XmlNode CreateOrGetFieldNodeByName(XmlDocument xmlDoc, string columnName, XmlNode fieldRoot)
		{
			XmlNode fieldNode = fieldRoot.SelectSingleNode(String.Format("//field[@name = '{0}']", columnName)); //???
			if (object.ReferenceEquals(fieldNode, null))
			{
				XmlElement newElement = xmlDoc.CreateElement("field");
				XmlAttribute newAttribute = xmlDoc.CreateAttribute("name");
				newAttribute.Value = columnName;
				newElement.Attributes.Append(newAttribute);
				fieldRoot.AppendChild(newElement);
				return newElement;
			}
			return fieldNode;
		}

		#endregion

		#region Mix

		public static string IndentXMLString_SAMPLE(string xml)
		{
			string outXml = string.Empty;
			MemoryStream ms = new MemoryStream();
			// Create a XMLTextWriter that will send its output to a memory stream (file)
			XmlTextWriter xtw = new XmlTextWriter(ms, Encoding.Unicode);
			XmlDocument doc = new XmlDocument();

			// Load the unformatted XML text string into an instance
			// of the XML Document Object Model (DOM)
			doc.LoadXml(xml);

			// Set the formatting property of the XML Text Writer to indented
			// the text writer is where the indenting will be performed
			xtw.Formatting = Formatting.Indented;

			// write dom xml to the xmltextwriter
			doc.WriteContentTo(xtw);
			// Flush the contents of the text writer
			// to the memory stream, which is simply a memory file
			xtw.Flush();

			// set to start of the memory stream (file)
			ms.Seek(0, SeekOrigin.Begin);
			// create a reader to read the contents of
			// the memory stream (file)
			StreamReader sr = new StreamReader(ms);
			// return the formatted string to caller
			return sr.ReadToEnd();
		}

		public static string IndentXMLString(XmlDocument pDoc)
		{
			string outXml = string.Empty;

			MemoryStream ms = new MemoryStream();
			XmlTextWriter xtw = new XmlTextWriter(ms, Encoding.Unicode);
			xtw.Formatting = Formatting.Indented;

			pDoc.WriteContentTo(xtw);
			xtw.Flush();

			ms.Seek(0, SeekOrigin.Begin);
			StreamReader sr = new StreamReader(ms);
			return sr.ReadToEnd();
		}

		public static string IndentXMLString(string xml)
		{
			XmlDocument doc = new XmlDocument();
			doc.LoadXml(xml);
			return IndentXMLString(doc);
		}

		public static void IndentXmlDocument(XmlDocument pDoc)
		{
			string ss = IndentXMLString(pDoc);
			pDoc.LoadXml(ss);
		}

		public static bool TryConvert<TValue>(string pValue, out TValue pNewValue)
		{
			if (String.IsNullOrEmpty(pValue))
			{
				pNewValue = default(TValue);
				return false;
			}

			Type type = typeof(TValue);

			if (type.IsEnum)
			{
				pNewValue = (TValue)Enum.Parse(type, pValue);
				return true;
			}

			//???
			//----------------------------
			switch (Type.GetTypeCode(type))
			{
				case TypeCode.Boolean:
					pNewValue = (TValue)(object)DbValueConvert.ToBool(pValue, false);
					return true;

				case TypeCode.Int16:
				case TypeCode.UInt16:
				case TypeCode.Int32:
				case TypeCode.UInt32:
				case TypeCode.Int64:
				case TypeCode.UInt64:
				case TypeCode.Decimal:
					pNewValue = (TValue)(object)DbValueConvert.ToInt(pValue, 0);
					return true;
			}

			//----------------------------
			// 20 aug 2010
			TypeConverter converter = TypeDescriptor.GetConverter(type);
			if (converter != null)
			{
				pNewValue = (TValue)converter.ConvertFromString(pValue);
				return true;
			}
			//----------------------------
			object value2 = Convert.ChangeType(pValue, type);
			pNewValue = (TValue)value2;
			//----------------------------
			return true;
		}

		#endregion

	}
}
